require("dotenv").config();
const db = require("../models");
const User = db.user;
const Image = db.image;
const Review = db.review;
const Op = db.Sequelize.Op;
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const otpGenerator = require("otp-generator");
const mailSender = require("../helper/mailSender");

exports.register = async (req, res, next) => {
  try {
    // Get user input
    let { name, surname, email, password } = req.body;

    // Validate user input
    if (!(name && surname && email && password)) {
      return res
        .status(200)
        .send({ status: false, message: "All input is required", data: null });
    }

    // if(password != confirmPassword) {
    //  return res.status(400).send({ status: "Passwords dont match!" });

    // }

    // check if user already exist
    // Validate if user exist in our database
    const oldUser = await User.findOne({ where: { email: email } });

    console.log(oldUser);

    if (oldUser != null) {
      return res.status(200).send({
        status: false,
        message: "User Already Exist. Please Login",
        data: null
      });
    }

    const otp = otpGenerator.generate(4, {
      upperCaseAlphabets: false,
      specialChars: false,
      lowerCaseAlphabets: false
    });
    console.log(otp);

    const mailResponse = await mailSender(
      email,
      "Verification Email",
      `<h1>Please confirm your OTP</h1>
       <p>Here is your OTP code: ${otp}</p>`
    );
    console.log("Email sent successfully: ", mailResponse);

    //Encrypt user password
    encryptedPassword = await bcrypt.hash(password, 10);

    // Create user in our database
    const user = {
      name: req.body.name,
      surname: req.body.surname,
      email: req.body.email,
      password: encryptedPassword,
      otp: otp
    };

    // Create token
    const token = jwt.sign({ user_id: user._id, email }, process.env.TOKEN_KEY);
    // save user token
    user.token = token;

    User.create(user)
      .then(data => {
        return res.status(200).send({
          status: true,
          message: "User created successfuly.",
          data: data
        });
      })
      .catch(err => {
        return res.status(500).send({
          message: err.message || "Some error occurred while creating the User."
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.login = async (req, res, next) => {
  // Our login logic starts here
  try {
    // Get user input
    let { email, password } = req.body;

    // Validate user input
    if (!(email && password)) {
      return res
        .status(200)
        .send({ status: false, message: "All input is required", data: null });
    }
    // Validate if user exist in our database
    const user = await User.findOne({ where: { email: email } });
    console.log(user);

    if (user.active == 0) {
      return res.status(200).send({
        status: false,
        message: "User needs to be activated.",
        data: null
      });
    }

    if (user && (await bcrypt.compare(password, user.password))) {
      // Create token
      const token = jwt.sign(
        { user_id: user._id, email },
        process.env.TOKEN_KEY
      );

      // save user token
      user.token = token;

      // user
      return res
        .status(200)
        .json({ status: true, message: "Login successful.", data: user });
    } else {
      return res
        .status(200)
        .json({ status: false, message: "Login fail.", data: null });
    }
  } catch (err) {
    console.log(err);
  }
};

exports.activateUser = async (req, res, next) => {
  try {
    let { email, otp } = req.body;

    if (!(email && otp)) {
      return res
        .status(200)
        .send({ status: false, message: "All input is required", data: null });
    }

    const oldUser = await User.findOne({ where: { email: email } });

    if (!oldUser) {
      return res
        .status(200)
        .send({ status: false, message: "User not found.", data: null });
    }

    if (oldUser.active === true) {
      return res.status(200).send({
        status: false,
        message: "User already activated.",
        data: null
      });
    }

    if (otp != oldUser.otp) {
      return res
        .status(200)
        .send({ status: false, message: "Otp code is not valid", data: null });
    }

    User.update(
      { active: true },
      {
        where: { email: email }
      }
    )
      .then(async num => {
        const updated = await User.findOne({
          where: { email: email },
          attributes: { exclude: ["token", "password", "otp", "active"] }
        });

        return res
          .status(200)
          .send({ status: true, message: "Updated", data: updated });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Error updating record",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.edit = async (req, res, next) => {
  // Our edit logic starts here
  try {
    const { id } = req.params;
    let { name, surename, phone, dateOfBirth, placeFrom } = req.body;

    const oldUser = await User.findOne({
      where: { id: id },
      include: [
        {
          model: Image,
          where: { user_id: id },
          required: false,
          exclude: ["token", "password", "otp", "active"]
        }
      ]
    });

    if (!oldUser) {
      return res
        .status(200)
        .send({ status: false, message: "User not found.", data: null });
    }
    console.log(name);

    if (name) oldUser.name = name;
    if (surename) oldUser.surename = surename;
    if (phone) oldUser.phone = phone;
    if (dateOfBirth) oldUser.dateOfBirth = dateOfBirth;
    if (placeFrom) oldUser.placeFrom = placeFrom;
    User.update(req.body, {
      where: { id: id }
    })
      .then(async num => {
        return res
          .status(200)
          .send({ status: true, message: "Updated", data: oldUser });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Error updating record",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.editImage = async (req, res, next) => {
  // Our edit logic starts here
  try {
    const { id } = req.params;
    let { imagePath } = req.body;

    const oldImage = await Image.findOne({
      where: { user_id: id }
    });

    if (!oldImage) {
      const img = {
        user_id: id,
        imagePath: imagePath
      };
      Image.create(img)
        .then(data => {
          return res.status(200).send({
            status: true,
            message: "Image created succesfully.",
            data: data
          });
        })
        .catch(err => {
          return res.status(200).send({
            status: false,
            message: err.message || "Faild.",
            data: null
          });
        });
    } else {
      if (imagePath) oldImage.imagePath = imagePath;

      Image.update(req.body, {
        where: { user_id: id }
      })
        .then(async num => {
          return res
            .status(200)
            .send({ status: true, message: "Updated", data: oldImage });
        })
        .catch(err => {
          return res.status(200).send({
            status: false,
            message: err.message || "Error updating record",
            data: null
          });
        });
    }
  } catch (err) {
    console.log(err);
  }
};

exports.getUserProfile = async (req, res, next) => {
  // Our login logic starts here
  try {
    const { user_id } = req.params;

    if (!user_id) {
      return res
        .status(200)
        .send({ status: false, message: "Missing params.", data: null });
    }

    await User.findByPk(user_id, {
      attributes: { exclude: ["token", "password", "otp", "active"] },
      include: [{ model: Image }, { model: Review }]
    })
      .then(data => {
        if (!data) {
          return res.status(200).send({
            status: false,
            message: "There is no user with that id!",
            data: null
          });
        }
        return res.status(200).send({
          status: true,
          message: "Successful!",
          data: data
        });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Faild.",
          data: null
        });
      });
    return next();
  } catch (err) {
    console.log(err);
  }
};

exports.savePushToken = async (req, res, next) => {
  try {
    const { user_id } = req.params;

    let { device_uuid, push_token } = req.body;

    if (!(device_uuid && push_token)) {
      return res
        .status(200)
        .send({ status: false, message: "All input is required", data: null });
    }

    const oldUser = await User.findOne({ where: { id: user_id } });

    if (!oldUser) {
      return res
        .status(200)
        .send({ status: false, message: "User not found.", data: null });
    }

    User.update(
      { device_uuid: device_uuid, push_token: push_token },
      {
        where: { id: user_id }
      }
    )
      .then(async num => {
        const updated = await User.findOne({
          where: { id: user_id },
          attributes: {
            exclude: [
              "token",
              "password",
              "otp",
              "active",
              "device_uuid",
              "push_token"
            ]
          }
        });

        return res
          .status(200)
          .send({ status: true, message: "Updated", data: updated });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Error updating record",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};
