require("dotenv").config();
const db = require("../models");
const Teleporter = db.teleporter;
const Sender = db.sender;
const User = db.user;
const Op = db.Sequelize.Op;
const Image = db.image;
const sequelize = require("sequelize");
const moment = require("moment");
const { paginate } = require("../helper/helper");

exports.create = async (req, res, next) => {
  try {
    let {
      user_id,
      name,
      departureLocation,
      destination,
      travelTime
    } = req.body;

    // Validate user input
    if (!(user_id && name && departureLocation && destination && travelTime)) {
      return res
        .status(200)
        .send({ status: false, message: "All input is required", data: null });
    }

    const user = await User.findOne({
      where: { id: user_id }
    });

    if (!user) {
      return res.status(200).send({
        status: false,
        message: "User with that id not found",
        data: null
      });
    }

    if (travelTime < moment(Date.now()).format("YYYY-MM-DD")) {
      return res.status(200).send({
        status: false,
        message: "Date cant be smaller then present date.",
        data: null
      });
    }

    const teleporter = {
      user_id: req.body.user_id,
      name: name,
      departureLocation: departureLocation,
      destination: destination,
      travelTime: moment(new Date(travelTime)).format("YYYY-MM-DD")
    };

    Teleporter.create(teleporter)
      .then(data => {
        return res.status(200).send({
          status: true,
          message: "Teleporter created successfuly.",
          data: data
        });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message:
            err.message || "Some error occurred while creating the teleporter.",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.edit = async (req, res, next) => {
  // Our edit logic starts here
  try {
    const { id } = req.params;
    let {
      user_id,
      name,
      available,
      departureLocation,
      destination,
      travelTime
    } = req.body;

    const oldTeleporter = await Teleporter.findOne({ where: { id: id } });

    if (!oldTeleporter) {
      return res
        .status(200)
        .send({ status: false, message: "Teleporter not found.", data: null });
    }

    if (user_id) oldTeleporter.user_id = user_id;
    if (name) oldTeleporter.name = name;
    if (available) oldTeleporter.available = available;
    if (departureLocation) oldTeleporter.departureLocation = departureLocation;
    if (destination) oldTeleporter.destination = destination;
    if (travelTime)
      oldTeleporter.travelTime = moment(new Date(travelTime)).format(
        "YYYY-MM-DD"
      );

    Teleporter.update(req.body, {
      where: { id: id }
    })
      .then(async num => {
        const updated = await Teleporter.findOne({ where: { id: id } });

        return res
          .status(200)
          .send({ status: true, message: "Updated", data: updated });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Error updating record",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.getAll = async (req, res, next) => {
  // Our getAll logic starts here
  try {
    const allTeleporters = await Teleporter.findAll();

    Teleporter.findAll()
      .then(data => {
        return res
          .status(200)
          .send({ status: true, message: "Successfuly fetched.", data: data });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message:
            err.message ||
            "Some error occurred while fetching the teleporters.",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.delete = async (req, res, next) => {
  // Our login logic starts here
  try {
    const { id } = req.params;

    const teleporter = await Teleporter.findOne({ where: { id: id } });

    if (!teleporter) {
      return res
        .status(200)
        .send({ status: false, message: "Teleporter not found", data: null });
    }

    Teleporter.destroy({ where: { id: id } })
      .then(data => {
        return res.status(200).send({
          status: true,
          message: "Successfuly deleted record.",
          data: teleporter
        });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Deleting faild.",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.getById = async (req, res, next) => {
  // Our login logic starts here
  try {
    const { id } = req.params;

    Teleporter.findOne({ where: { id: id } })
      .then(data => {
        if (!data) {
          return res.status(200).send({
            status: false,
            message: "Teleporter not found",
            data: null
          });
        }
        return res
          .status(200)
          .send({ status: true, message: "Successfuly fetched.", data: data });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Faild.",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.searchForTraveler = async (req, res, next) => {
  try {
    const { page, limit } = req.params;
    let { departure, destination, time } = req.body;

    if (!(departure && destination && time)) {
      return res
        .status(200)
        .send({ status: false, message: "All input is required", data: null });
    }

    if (time < moment(Date.now()).format("YYYY-MM-DD")) {
      return res.status(200).send({
        status: false,
        message: "Date cant be smaller then present date.",
        data: null
      });
    }

    Teleporter.findAll({
      where: {
        departureLocation: departure,
        destination: destination,
        available: true,
        travelTime: { [Op.gte]: time }
      },
      include: [
        {
          model: User,
          attributes: {
            exclude: [
              "token",
              "password",
              "otp",
              "active",
              "device_uuid",
              "push_token"
            ]
          },

          include: [{ model: Image }]
        }
      ],
      order: [["createdAt", "DESC"]],
      ...paginate({ currentPage: page, pageSize: limit })
    })
      .then(data => {
        if (!data || data.length == 0) {
          return res.status(200).send({
            status: false,
            message: "There is no available teleporters for this package!",
            data: null
          });
        }

        return res
          .status(200)
          .send({ status: true, message: "Successfuly fetched!", data: data });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Faild.",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.search = async (req, res, next) => {
  try {
    const { user_id, page, limit } = req.params;
    let { searchTerm } = req.body;

    if (!searchTerm) {
      return res.status(200).send({
        status: false,
        message: "Input is required.",
        data: null
      });
    }

    if (searchTerm) {
      await Teleporter.findAndCountAll({
        where: [
          {
            [Op.or]: [
              { name: { [Op.like]: `%${searchTerm}%` } },
              { departureLocation: { [Op.like]: `%${searchTerm}%` } },
              { destination: { [Op.like]: `%${searchTerm}%` } },
              { travelTime: { [Op.like]: `%${searchTerm}%` } }
            ]
          },
          { user_id: { [Op.not]: user_id } }
        ],
        include: [
          {
            model: User,
            attributes: { exclude: ["token", "password", "otp", "active"] },
            include: [{ model: Image }]
          }
        ],
        distinct: true,
        order: [["createdAt", "DESC"]],
        ...paginate({ currentPage: page, pageSize: limit })
      })
        .then(data => {
          if (!data["rows"] || data["rows"].length == 0) {
            return res.status(200).send({
              status: false,
              message: "There is no available teleporters for this search!",
              data: null
            });
          }
          return res.status(200).send({
            status: true,
            message: "Successfuly fetched!",
            data: data
          });
        })
        .catch(err => {
          return res.status(200).send({
            status: false,
            message: err.message || "Faild.",
            data: null
          });
        });
      return next();
    }
  } catch (err) {
    console.log(err);
  }
};

exports.activeTeleports = async (req, res, next) => {
  // Our login logic starts here
  try {
    const { teleporterId, page, limit } = req.params;

    if (!teleporterId) {
      return res
        .status(200)
        .send({ status: false, message: "Missing teleporterId.", data: null });
    }

    await Sender.findAndCountAll({
      where: {
        teleporter_id: teleporterId,
        status: "Teleporting"
      },
      include: Image,
      distinct: true,
      order: [["createdAt", "DESC"]],
      ...paginate({ currentPage: page, pageSize: limit })
    })
      .then(data => {
        if (!data["rows"] || data["rows"].length == 0) {
          return res.status(200).send({
            status: false,
            message: "There is no available teleports!",
            data: null
          });
        }

        return res
          .status(200)
          .send({ status: true, message: "Successfuly fetched!", data: data });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Faild.",
          data: null
        });
      });
    return next();
  } catch (err) {
    console.log(err);
  }
};

exports.historyTeleports = async (req, res, next) => {
  // Our login logic starts here
  try {
    const { userId, page, limit } = req.params;

    if (!userId) {
      return res
        .status(200)
        .send({ status: false, message: "Missing userId.", data: null });
    }

    await Sender.findAndCountAll({
      where: {
        status: "Complited",
        teleporter_id: userId
      },
      include: Image,
      distinct: true,
      order: [["createdAt", "DESC"]],
      ...paginate({ currentPage: page, pageSize: limit })
    })
      .then(data => {
        if (!data["rows"] || data["rows"].length == 0) {
          return res.status(200).send({
            status: false,
            message: "There is no available history!",
            data: null
          });
        }

        return res
          .status(200)
          .send({ status: true, message: "Successfuly fetched!", data: data });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Faild.",
          data: null
        });
      });
    return next();
  } catch (err) {
    console.log(err);
  }
};
